import React from 'react';

// --- REMOTION MOCK SYSTEM (for Canvas preview) ---
// Corrected Easing and Interpolation to support multi-point ranges and standard math
const Easing = {
  out: (f) => (t) => 1 - f(1 - t),
  in: (f) => (t) => f(t),
  inOut: (f) => (t) => t < 0.5 ? f(2 * t) / 2 : (2 - f(2 * (1 - t))) / 2,
  linear: (t) => t,
  quad: (t) => t * t,
  cubic: (t) => t * t * t,
  bezier: (x1, y1, x2, y2) => (t) => {
    const cx = 3 * x1;
    const bx = 3 * (x2 - x1) - cx;
    const ax = 1 - cx - bx;
    const cy = 3 * y1;
    const by = 3 * (y2 - y1) - cy;
    const ay = 1 - cy - by;
    const solveX = (x) => {
      let t2 = x;
      for (let i = 0; i < 8; i++) {
        const x2 = ((ax * t2 + bx) * t2 + cx) * t2 - x;
        if (Math.abs(x2) < 0.001) return t2;
        const d2 = (3 * ax * t2 + 2 * bx) * t2 + cx;
        if (Math.abs(d2) < 1e-6) break;
        t2 -= x2 / d2;
      }
      return t2;
    };
    const t_ = solveX(t);
    return ((ay * t_ + by) * t_ + cy) * t_;
  },
  elastic: (bounciness = 1) => (t) => {
    const p = bounciness * Math.PI;
    return 1 - Math.pow(Math.cos((t * Math.PI) / 2), 3) * Math.cos(t * p);
  }
};

const interpolate = (input, inputRange, outputRange, options = {}) => {
  const { extrapolateLeft = 'extend', extrapolateRight = 'extend', easing } = options;

  if (inputRange.length !== outputRange.length) {
    throw new Error('inputRange and outputRange must have the same length');
  }

  // Handle single-point edge cases
  if (inputRange.length === 1) return outputRange[0];

  // Find the current segment
  let i = 0;
  for (; i < inputRange.length - 1; i++) {
    if (input < inputRange[i + 1]) break;
  }

  // Handle extrapolation
  if (input <= inputRange[0]) {
    if (extrapolateLeft === 'clamp') return outputRange[0];
    const slope = (outputRange[1] - outputRange[0]) / (inputRange[1] - inputRange[0]);
    return outputRange[0] + (input - inputRange[0]) * slope;
  }

  if (input >= inputRange[inputRange.length - 1]) {
    if (extrapolateRight === 'clamp') return outputRange[outputRange.length - 1];
    const last = inputRange.length - 1;
    const slope = (outputRange[last] - outputRange[last - 1]) / (inputRange[last] - inputRange[last - 1]);
    return outputRange[last] + (input - inputRange[last]) * slope;
  }

  const minInput = inputRange[i];
  const maxInput = inputRange[i + 1];
  const minOutput = outputRange[i];
  const maxOutput = outputRange[i + 1];

  let progress = (input - minInput) / (maxInput - minInput);
  if (easing) {
    // If multiple segments, apply easing per segment
    progress = easing(progress);
  }
  
  return minOutput + progress * (maxOutput - minOutput);
};

const useCurrentFrame = () => {
  const [frame, setFrame] = React.useState(0);
  React.useEffect(() => {
    const interval = setInterval(() => setFrame(f => (f + 1) % 300), 1000 / 30);
    return () => clearInterval(interval);
  }, []);
  return frame;
};

const useVideoConfig = () => ({ 
  width: 1920, 
  height: 1080, 
  fps: 30, 
  durationInFrames: 300 
});

const AbsoluteFill = ({ children, style }) => (
  <div style={{ position: 'absolute', top: 0, left: 0, right: 0, bottom: 0, ...style }}>
    {children}
  </div>
);

// =============================================================================
// PRE-GENERATED DATA
// =============================================================================

const seededRandom = (seed) => {
  const x = Math.sin(seed * 9999) * 10000;
  return x - Math.floor(x);
};

const particles = Array.from({ length: 40 }).map((_, i) => ({
  x: seededRandom(i) * 100,
  y: seededRandom(i + 100) * 100,
  size: seededRandom(i + 200) * 3 + 1,
  speed: seededRandom(i + 300) * 0.5 + 0.1,
  opacity: seededRandom(i + 400) * 0.4 + 0.1,
}));

// =============================================================================
// MAIN COMPONENT
// =============================================================================

const MoneyCounter = () => {
  const frame = useCurrentFrame();
  const { width, height } = useVideoConfig();

  // 1. Ticking Logic (0 to 1000 over 150 frames)
  const count = interpolate(frame, [0, 150], [0, 1000], {
    extrapolateLeft: 'clamp',
    extrapolateRight: 'clamp',
    easing: Easing.out(Easing.quad),
  });

  // 2. Growth Logic (at frame 150)
  // Now supports multi-point interpolation correctly (1.0 -> 1.35 -> 1.2)
  const scale = interpolate(frame, [150, 165, 185], [1, 1.35, 1.2], {
    extrapolateLeft: 'clamp',
    extrapolateRight: 'clamp',
    easing: Easing.bezier(0.34, 1.56, 0.64, 1),
  });

  // 3. Counter "Flash" (Glow intensity burst)
  const glowBurst = interpolate(frame, [150, 155, 180], [10, 80, 20], {
    extrapolateLeft: 'clamp',
    extrapolateRight: 'clamp',
  });

  // 4. Infinite Idle Pulse (after the growth)
  const pulse = Math.sin(frame / 12) * 0.015 + 1;
  const finalScale = frame > 185 ? scale * pulse : scale;

  return (
    <AbsoluteFill style={{ backgroundColor: '#020617', overflow: 'hidden', fontFamily: 'system-ui, sans-serif' }}>
      
      {/* Dynamic Background Glow */}
      <div style={{
        position: 'absolute',
        width: '100%',
        height: '100%',
        background: `radial-gradient(circle at 50% 50%, rgba(16, 185, 129, ${interpolate(frame, [150, 160], [0.1, 0.25], { extrapolateRight: 'clamp' })}) 0%, transparent 70%)`,
      }} />

      {/* Floating Particles */}
      {particles.map((p, i) => {
        const moveY = (frame * p.speed) % 110;
        return (
          <div
            key={i}
            style={{
              position: 'absolute',
              left: `${p.x}%`,
              top: `${(p.y - moveY + 110) % 110 - 5}%`,
              width: p.size,
              height: p.size,
              backgroundColor: '#10b981',
              borderRadius: '50%',
              opacity: p.opacity,
              boxShadow: '0 0 8px #10b981',
            }}
          />
        );
      })}

      {/* Main Counter Container */}
      <div style={{
        position: 'absolute',
        top: '50%',
        left: '50%',
        transform: `translate(-50%, -50%) scale(${finalScale})`,
        textAlign: 'center',
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
      }}>
        
        {/* Money Value Display */}
        <div style={{
          fontSize: '280px',
          fontWeight: '900',
          color: '#f8fafc',
          display: 'flex',
          alignItems: 'center',
          filter: `drop-shadow(0 0 ${glowBurst}px rgba(16, 185, 129, 0.8))`,
        }}>
          <span style={{ color: '#10b981', marginRight: '20px' }}>$</span>
          <span style={{ fontVariantNumeric: 'tabular-nums', letterSpacing: '-5px' }}>
            {Math.floor(count).toLocaleString()}
          </span>
        </div>

      </div>

    </AbsoluteFill>
  );
};

export default MoneyCounter;

export const RemotionVideo = {
  component: MoneyCounter,
  durationInFrames: 300,
  fps: 30,
  width: 1920,
  height: 1080,
  id: 'MoneyCounter',
};